<?php
/**
 * HealthRiskScorer.php
 * Calculates member health risk scores based on demographics, claims, and clinical data.
 */
class HealthRiskScorer
{
    private $conn;

    public function __construct($conn)
    {
        $this->conn = $conn;
    }

    /**
     * Calculate and save risk score for a member
     * 
     * @param int $memberId
     * @return array Result with score and factors
     */
    public function calculateMemberRisk($memberId)
    {
        $riskScore = 0;
        $factors = [];

        // 1. Demographic Risk (Age)
        $ageSql = "SELECT dob, gender FROM members WHERE member_id = ?";
        $stmt = $this->conn->prepare($ageSql);
        $stmt->bind_param("i", $memberId);
        $stmt->execute();
        $memberData = $stmt->get_result()->fetch_assoc();

        if ($memberData) {
            $dob = new DateTime($memberData['dob']);
            $now = new DateTime();
            $age = $now->diff($dob)->y;

            if ($age > 60) {
                $riskScore += 20;
                $factors[] = "Age > 60 (+20)";
            } elseif ($age > 45) {
                $riskScore += 10;
                $factors[] = "Age > 45 (+10)";
            }
        }

        // 2. Chronic Condition Risk (Based on ICD-10 history in claims)
        // Adjust table names based on your actual schema (assuming `claims` or `bills` and `icd10`)
        $chronicSql = "SELECT DISTINCT b.diagnosis 
                       FROM bills b 
                       WHERE b.member = ? 
                       AND b.created_at >= DATE_SUB(NOW(), INTERVAL 2 YEAR)";
        // Note: This is a simplified check. Real systems would map ICD-10 codes to chronic categories
        $stmt = $this->conn->prepare($chronicSql);
        $stmt->bind_param("i", $memberId);
        $stmt->execute();
        $codeResult = $stmt->get_result();

        $chronicFound = false;
        while ($row = $codeResult->fetch_assoc()) {
            // Simplified detection of chronic triggers (Diabetes, Hypertension, etc.)
            $diag = strtoupper($row['diagnosis'] ?? '');
            if (strpos($diag, 'E11') !== false || strpos($diag, 'I10') !== false) {
                $riskScore += 40;
                $factors[] = "Chronic Condition Detected ($diag) (+40)";
                $chronicFound = true;
                break; // Cap at one major chronic flag for this simple engine
            }
        }

        // 3. Utilization Risk (Frequency)
        $utilSql = "SELECT COUNT(*) as visit_count FROM bills WHERE member = ? AND created_at >= DATE_SUB(NOW(), INTERVAL 1 YEAR)";
        $stmt = $this->conn->prepare($utilSql);
        $stmt->bind_param("i", $memberId);
        $stmt->execute();
        $utilData = $stmt->get_result()->fetch_assoc();

        if ($utilData['visit_count'] > 12) { // More than 1 visit per month on avg
            $riskScore += 25;
            $factors[] = "High Utilization (>12 visits/yr) (+25)";
        }

        // Cap score
        $riskScore = min(100, $riskScore);

        // Save to DB
        $this->saveRiskScore($memberId, $riskScore, $factors);

        return [
            'member_id' => $memberId,
            'score' => $riskScore,
            'factors' => $factors
        ];
    }

    private function saveRiskScore($memberId, $score, $factors)
    {
        $jsonFactors = json_encode($factors);
        $sql = "INSERT INTO member_risk_scores (member_id, risk_score, risk_factors_json, last_updated) 
                VALUES (?, ?, ?, NOW()) 
                ON DUPLICATE KEY UPDATE risk_score = VALUES(risk_score), risk_factors_json = VALUES(risk_factors_json), last_updated = NOW()";

        $stmt = $this->conn->prepare($sql);
        $stmt->bind_param("iis", $memberId, $score, $jsonFactors);
        $stmt->execute();
    }
}
?>